#include "StateMachine.h"
#include "HTMLWaiterReactor.h"
#include "SummaryReactor.h"
#include "GeneralHTMLReactor.h"
#include "Base64.h"
#include "Grabbag.h"
#include "GUI/MiniGUI.h"
#include <fstream>
using namespace std;

namespace {
    class ReactorGUI: public ProblemHandler {
    public:
        ReactorGUI(GWindow& window);

        void hyperlinkClicked(const string& url) override;

    private:
        class GraphicsIO: public IODevice {
        public:
            GraphicsIO(ReactorGUI* owner);

            void clear() override;
            void outputText(const std::string& text) override;
            void outputHeader(const std::string& text) override;
            void outputLink(int index, const std::string& text) override;
            void outputNext() override;
            std::string getLine(const std::string& prompt, const std::string& title, const std::string& defaultValue) override;
            bool getYesOrNo(const std::string& prompt, const std::string& title) override;
            void flush() override;

        private:
            ReactorGUI* owner;
            shared_ptr<stringstream> buffer;
        };

        Temporary<GBrowserPane> pane;
        shared_ptr<GraphicsIO> io;
        shared_ptr<StateMachine> stateMachine;
    };

    /* Constant representing the comment to look for when injecting a new piece of content. */
    const string kInjectionSite   = "<!-- Inject ";
    const string kInjectionCloser = "-->";

    /* Constant representing the comment to look for when injecting a base64-encoded piece
     * of content.
     */
    const string kBase64InjectionSite   = "<!-- Base64Inject ";
    const string kBase64InjectionCloser = "-->";

    /* Constructs the graphics system. */
    shared_ptr<GraphicsSystem> makeGraphics(IODevice* io) {
        shared_ptr<GraphicsSystem> result = make_shared<GraphicsSystem>();
        result->io = io;
        return result;
    }

    /* Given a string, replaces all injection sites with the appropriate contents. */
    string replaceInjectionSitesIn(const string& str, const Grabbag& grabbag) {
        string result;

        size_t index = 0;
        while (true) {
            /* See if we can find an injection site. */
            size_t site = str.find(kInjectionSite, index);

            /* If there isn't one, then whatever we've assembled so far, plus the rest
             * of the string, is the final result.
             */
            if (site == string::npos) return result + str.substr(index);

            /* Otherwise, we found an injection site. Everything up to this point is
             * perfectly safe to add in.
             */
            result += string(str.begin() + index, str.begin() + site);

            /* Otherwise, we found a site. See what to replace it with. */
            size_t endpoint = str.find(kInjectionCloser, site);
            if (endpoint == string::npos) error("Unterminated injection site?");

            /* Get the filename from this range. */
            string filename(str.begin() + site + kInjectionSite.size(),
                            str.begin() + endpoint);

            /* Recursively replace injections within that file and append that to the result. */
            result += replaceInjectionSitesIn(grabbag.contentsOf(trim(filename)), grabbag);

            /* Scoot past this point in the file. */
            index = endpoint + kInjectionCloser.size();
        }
    }

    /* Given a string, replaces all base64 injection sites with the appropriate contents. */
    string replaceBase64In(const string& str, const Grabbag& grabbag) {
        string result;

        size_t index = 0;
        while (true) {
            /* See if we can find an injection site. */
            size_t site = str.find(kBase64InjectionSite, index);

            /* If there isn't one, then whatever we've assembled so far, plus the rest
             * of the string, is the final result.
             */
            if (site == string::npos) return result + str.substr(index);

            /* Otherwise, we found an injection site. Everything up to this point is
             * perfectly safe to add in.
             */
            result += string(str.begin() + index, str.begin() + site);

            /* Determine what to replace things with. */
            size_t endpoint = str.find(kBase64InjectionCloser, site);
            if (endpoint == string::npos) error("Unterminated injection site?");

            /* Get the filename from this range. */
            string filename(str.begin() + site + kBase64InjectionSite.size(),
                            str.begin() + endpoint);

            /* Base64-encode the contents of that file. */
            result += Base64::encode(grabbag.contentsOf(trim(filename)));

            /* Scoot past this point in the file. */
            index = endpoint + kBase64InjectionCloser.size();
        }
    }

    /* Data sourcing function for a Grabbag. */
    StateReader grabbagReader(const string& grabbagFile) {
        ifstream input(grabbagFile, ios::binary);
        if (!input) error("Cannot open grabbag file " + grabbagFile);

        Grabbag grabbag(input);

        return [grabbag](const string& filename) {
            string text = grabbag.contentsOf("states/" + filename + ".state");

            /* TODO: With C++14 support, use make_unique. */
            return unique_ptr<istringstream>(new istringstream(replaceBase64In(replaceInjectionSitesIn(text, grabbag), grabbag)));
        };
    }

    shared_ptr<StateMachine> createStateMachine(IODevice* io) {
        StateMachineBuilder builder(makeGraphics(io), "Welcome", grabbagReader("res/assignment.grabbag"));

        HTMLWaiterReactor::installHandlers(builder);
        GeneralHTMLReactor::installHandlers(builder);
        SummaryReactor::installHandlers(builder);

        return builder.build();
    }

    ReactorGUI::ReactorGUI(GWindow& window): ProblemHandler(window) {
        pane = make_temporary<GBrowserPane>(window, "CENTER");
        io   = make_shared<GraphicsIO>(this);
        stateMachine = createStateMachine(io.get());
    }

    void ReactorGUI::hyperlinkClicked(const string& url) {
        stateMachine->handleEvent(url);
    }

    const string kStyle = R"(<style type="text/css">
  body {
    font-family: serif;
    font-size: 24pt;
    color: black;
    background-color: white;
  }

  h1 {
    font-size: 30pt;
  }

  a {
    font-weight: bold;
  }
</style>)";

    ReactorGUI::GraphicsIO::GraphicsIO(ReactorGUI* owner) : owner(owner) {

    }

    void ReactorGUI::GraphicsIO::clear() {
        buffer = make_shared<stringstream>();
        *buffer << "<html><head>" << kStyle << "</head><body>";
    }
    void ReactorGUI::GraphicsIO::outputText(const std::string& text) {
        *buffer << "<p>" << text << "</p>";
    }
    void ReactorGUI::GraphicsIO::outputHeader(const std::string& text) {
        *buffer << "<h1>" << text << "</h1>";
    }
    void ReactorGUI::GraphicsIO::outputLink(int index, const std::string& text) {
        *buffer << "<p><a href=\"" << index << "\">" << text << "</a></p>";
    }
    void ReactorGUI::GraphicsIO::outputNext() {
        *buffer << "<p><a href=\"next\">Click here to continue.</a><br></p>";
    }

    string ReactorGUI::GraphicsIO::getLine(const string& prompt, const string& title, const string& defaultValue) {
        return GOptionPane::showInputDialog(&owner->window(), prompt, title, defaultValue);
    }
    bool ReactorGUI::GraphicsIO::getYesOrNo(const std::string& prompt, const string& title) {
        return GOptionPane::showConfirmDialog(&owner->window(), prompt, title) == GOptionPane::CONFIRM_YES;
    }

    void ReactorGUI::GraphicsIO::flush() {
        *buffer << "</body></html>";
        owner->pane->readTextFromFile(*buffer);
    }

    /* Console IO device. */
    class ConsoleIO: public IODevice {
    public:
        ConsoleIO(Vector<string>& options, bool& isNext) : options(options), isNext(isNext) {}

        void clear() override {
            options.clear();
            isNext = false;
        }
        void outputText(const string& text) override {
            cout << text << endl << endl;
        }
        void outputHeader(const string& text) override {
            cout << text << endl << endl;
        }
        void outputLink(int, const std::string& text) override {
            options.add(text);
        }
        void outputNext() override {
            isNext = true;
        }
        string getLine(const string& prompt, const string&, const string&) override {
            return ::getLine(prompt);
        }
        bool getYesOrNo(const string& prompt, const string&) override {
            return ::getYesOrNo(prompt);
        }
        void flush() override {
            // Has no effect.
        }

    private:
        Vector<string>& options;
        bool& isNext;
    };
}

GRAPHICS_HANDLER("Problem Set Zero", GWindow& window) {
    return make_shared<ReactorGUI>(window);
}

CONSOLE_HANDLER("Problem Set Zero") {
    Vector<string> options;
    bool isNext = false;

    auto io = make_shared<ConsoleIO>(options, isNext);
    auto stateMachine = createStateMachine(io.get());

    while (true) {
        /* We might have a "next" prompt. */
        if (isNext) {
            (void) getLine("Press ENTER to continue.");
            stateMachine->handleEvent("next");
        }
        /* Otherwise, make a selection. */
        else if (!options.isEmpty()){
            /* Add one because makeSelectionFrom is zero-indexed. */
            string choice = to_string(makeSelectionFrom("Choose an answer:", options) + 1);
            stateMachine->handleEvent(choice);
        } else {
            cout << "That's it! You're all set. You may close this window." << endl;
            while (true) {
                (void) getLine();
            }
        }
    }
}
